#!/bin/bash
# C Code Run (C) 2016-2017 Stephane Charette <stephanecharette@gmail.com>
# $Id: ccr_user.sh 2205 2017-04-16 08:23:43Z stephane $
#
# https://www.ccoderun.ca/bbtools/
#
# Create a user account and lock down root.


# must be root to create users and set passwords
if (( EUID != 0 )); then
	logger --id --priority user.error --stderr "$0: must be root to modify users"
	exit 1
fi


# what username/password should we create?
backup_username="ccr"
backup_password="icx100"
if [ -e /opt/ccr/files/user.ccr ]; then
	. /opt/ccr/files/user.ccr
fi
if [ "$1" != "" ]; then
	backup_username="$1"
fi
if [ "$2" != "" ]; then
	backup_password="$2"
fi
echo "backup_username='${backup_username}'" >  /opt/ccr/files/user.ccr
echo "backup_password='${backup_password}'" >> /opt/ccr/files/user.ccr
chmod 600 /opt/ccr/files/user.ccr


# lock down sshd
grep --quiet "PermitEmptyPasswords yes" /etc/ssh/sshd_config
if [ $? -eq 0 ]; then
	logger --id --priority user.notice --stderr "$0: locking down some open permissions in sshd"
	sed --in-place --expression 's:UsePAM no:UsePAM yes:g'								/etc/ssh/sshd_config
	sed --in-place --expression 's:PermitRootLogin yes:PermitRootLogin no:g'			/etc/ssh/sshd_config
	sed --in-place --expression 's:PermitEmptyPasswords yes:PermitEmptyPasswords no:g'	/etc/ssh/sshd_config
	sed --in-place --expression 's:PrintLastLog yes:PrintLastLog no:g'					/etc/ssh/sshd_config

	systemctl restart ssh
fi


# see if we need to lock down one of the default usernames
for name in root debian ubuntu; do

	getent passwd ${name} > /dev/null
	if [ $? -eq 0 ]; then

		if [ "${name}" = "${backup_username}" ]; then
			# do not disable the account we want to use
			continue
		fi

		passwd --status ${name} | grep --quiet "${name} L "
		if [ $? -ne 0 ]; then
			logger --id --priority user.notice --stderr "$0: locking down user '${name}'"
			chpasswd <<< "${name}:${backup_password}"
			passwd --lock ${name} > /dev/null
		fi
	fi
done


# create the backup user
getent passwd ${backup_username} > /dev/null
if [ $? -ne 0 ]; then
	logger --id --priority user.notice --stderr "$0: creating backup account '${backup_username}'"
	useradd --create-home --shell /bin/bash --user-group --groups sudo ${backup_username}
	chpasswd <<< "${backup_username}:${backup_password}"
fi
alias_filename=/home/${backup_username}/.bash_aliases
if [ ! -e ${alias_filename} ]; then
	echo "alias dir='ls -lh --color=auto'" > ${alias_filename}
	chown ${backup_username}:${backup_username} ${alias_filename}
fi


# replace the existing user motd and login issue messages
cp -f /opt/ccr/files/motd.ccr		/etc/motd
cp -f /opt/ccr/files/issue.ccr		/etc/issue
cp -f /opt/ccr/files/issue.net.ccr	/etc/issue.net


# check various accounts to see if they're still using the default password
for name in ccr root debian ubuntu ${backup_username}; do

	getent passwd ${name} > /dev/null
	if [ $? -eq 0 ]; then
		ccr_default_user_passwd ${name}
		if [ $? -ne 0 ]; then
			if [ -x /usr/bin/smk ]; then
				smk unique 1 "User account ${name} is still using the default password."
			else
				echo -e "\e[33;1mWARNING: User account ${name} is still using the default password.\nPlease remember to change the password to secure the device.\e[0m\n" >> /etc/motd
			fi
		fi
	fi
done

exit 0
