#!/bin/bash
# C Code Run (C) 2016-2017 Stephane Charette <stephanecharette@gmail.com>
# $Id: ccr_wifi_connection.sh 2116 2017-01-18 13:13:47Z stephane $
#
# https://www.ccoderun.ca/bbtools/
#
# Configure wifi so it connects to the given SSID.
# 1st parm is the ssid, 2nd parm is the passphrase.


interface="wlan0"
ssid_name="CCodeRun"
passphrase="testing"
connman_dir="/var/lib/connman"


# must be root to change wifi configuration
if (( EUID != 0 )); then
	logger --id --priority user.error --stderr "$0: must be root to change wifi configuration"
	exit 1
fi


# read the wifi name and passphrase from the command-line or from wifi.ccr
mkdir -p /opt/ccr/files
if [ -e /opt/ccr/files/wifi.ccr ]; then
	. /opt/ccr/files/wifi.ccr
fi
if [ "$1" != "" ]; then
	ssid_name="$1"
fi
if [ "$2" != "" ]; then
	passphrase="$2"
fi
echo "ssid_name='${ssid_name}'"		>  /opt/ccr/files/wifi.ccr
echo "passphrase='${passphrase}'"	>> /opt/ccr/files/wifi.ccr
echo "interface='${interface}'"		>> /opt/ccr/files/wifi.ccr
echo "connman_dir='${connman_dir}'"	>> /opt/ccr/files/wifi.ccr
chmod 600 /opt/ccr/files/wifi.ccr


# check to see if the connman directory has been created
if [ ! -d "${connman_dir}" ]; then
	logger --id --priority user.warning --stderr "$0: connman directory ${connman_dir} does not exist"
	# wait a short period of time before we try again
	sleep 30
	# restart the script (in case a new version was installed)
	exec $0
fi


# try several times to see if this device has a wifi adapter
has_wifi=0
for i in {1..5}
do
	iwconfig ${interface} | grep --quiet "802.11"
	if [ $? -eq 0 ]; then
		has_wifi=1
		break;
	fi
	# try again
	sleep 30
done

if [ ${has_wifi} = 0 ]; then
	# looks like this device does NOT have a wifi adapter
	logger --id --priority user.warning --stderr "$0: wifi adapter ${interface} not detected"
	exit 0
fi


ssid_hex=$(echo -n "${ssid_name}" | xxd -g0 | cut -d' ' -f2)
hw_addr="$(cat /sys/class/net/${interface}/address)"
hw_addr="${hw_addr//:}"

if [ -z "${hw_addr}" ]; then
	logger --id --priority user.warning --stderr "$0: failed to find the hardware address for ${interface}"
	# wait a short period of time before we try again
	sleep 30
	# restart the script (in case a new version was installed)
	exec $0
fi

service_name="wifi_${hw_addr}_${ssid_hex}_managed_psk"
config_filename="${connman_dir}/wifi.config"

recreate_file=0
if [ ! -e "${config_filename}" ]; then
	recreate_file=1
else
	grep --quiet "${passphrase}" ${config_filename}
	if [ $? -ne 0 ]; then
		recreate_file=1
	fi
	grep --quiet "${ssid_name}" ${config_filename}
	if [ $? -ne 0 ]; then
		recreate_file=1
	fi
fi

if [ ${recreate_file} -ne 0 ]; then
	logger --id --priority user.notice --stderr "$0: configuring ${wlan0} to connect to ${ssid_name}"

	cat <<- EOF > "${config_filename}"
	[service_${service_name}]
	Type=wifi
	Name=${ssid_name}
	SSID=${ssid_hex}
	Favorite=true
	AutoConnect=true
	Passphrase=${passphrase}
	Nameservers=8.8.8.8
	EOF
	#8.8.8.8 8.8.4.4 2001:4860:4860::8888 2001:4860:4860::8844

	chmod 600 "${config_filename}"

	connmanctl disable p2p
	connmanctl disable bluetooth
	connmanctl disable offline
	connmanctl tether wifi disable
	connmanctl enable wifi
	connmanctl connect ${service_name}
fi

exit 0
