/* JPEGWrap (C) 2017 Stephane Charette <stephanecharette@gmail.com>
 * $Id: JWCompress.hpp 2139 2017-02-02 11:17:46Z stephane $
 */


#pragma once

#include "JW.hpp"


/// Use %JWCompress to create @p .jpg files using RGB or YUV image data.
class JWCompress : public JW
{
	public:

		/// Constructor.
		JWCompress( void );

		/// Destructor.
		virtual ~JWCompress( void );

		/// Compress the given data.
		virtual JWCompress &compress( void );

		/** Create a JPEG from 16-bit YUYV data (aka YUY2, or YUV 4:2:2 format).  This format is popular with web cams.
		 * The output file is closed in this method once the image has been written.
		 * @see https://linuxtv.org/downloads/v4l-dvb-apis-new/uapi/v4l/pixfmt-yuyv.html
		 * @see https://www.fourcc.org/pixel-format/yuv-yuy2/
		 */
		virtual JWCompress &compress_yuv422( const uint8_t * const data, const size_t bytes_per_row  );

		/// Set the output file into which the compressed JPEG image will be stored. @{
		inline virtual JWCompress &set_output_file( const std::string &filename )	{ set_file( filename );	return *this;	}
		inline virtual JWCompress &set_output_file( FILE * f )						{ set_file( f );		return *this;	}
		/// @}

		/// Set the size of the image.
		virtual JWCompress &set_size( const size_t width, const size_t height );

		/// Set the number of colour components and the colour space.
		virtual JWCompress &set_colour( const int components = 3, const J_COLOR_SPACE colour_space = JCS_RGB );

		/** Set usable default values for most of the optional parameters.
		 * This sets all JPEG parameters to reasonable defaults, using only the input image's color space.  Many
		 * applications will only need to use this routine and perhaps @ref set_quality().
		 * @note This is automatically called by @ref JWCompress::JWCompress().
		 */
		virtual JWCompress &set_default_parameters( const J_COLOR_SPACE colour_space = JCS_RGB );

		/** Constructs JPEG quantization tables appropriate for the indicated quality setting.  The quality value is
		 * expressed on the 0..100 scale recommended by IJG.  The exact mapping from quality values to tables may
		 * change in future IJG releases as more is learned about DCT quantization.  If the force_baseline parameter is
		 * @p TRUE, then the quantization table entries are constrained to the range 1..255 for full JPEG baseline
		 * compatibility.  In the current implementation, this only makes a difference for quality settings below 25,
		 * and it effectively prevents very small/low quality files from being generated.  The IJG decoder is capable
		 * of reading the non-baseline files generated at low quality settings when force_baseline is @p FALSE, but
		 * other decoders may not be.
		 */
		virtual JWCompress &set_quality( const int quality = 75, const bool force_baseline = true );

		/// Arithmetic coding will usually result in marginally smaller files, but require more CPU than Huffman coding. @{
		inline virtual JWCompress &use_arithmetic_coding( void )	{ cinfo.arith_code = true;	return *this; }
		inline virtual JWCompress &use_huffman_coding( void )		{ cinfo.arith_code = false;	return *this; }
		/// @}

		/** Causes the compressor to compute optimal Huffman coding tables for the image.  This requires an extra pass
		 * over the data and therefore costs a good deal of space and time.  The default is @p FALSE, which tells the
		 * compressor to use the supplied or default Huffman tables.  In most cases optimal tables save only a few
		 * percent of file size compared to the default tables.  @see @ref use_huffman_coding()
		 */
		inline virtual JWCompress &optimized_huffman_tables( void )	{ cinfo.optimize_coding = true; return *this; }

		/** The FLOAT method is very slightly more accurate than the ISLOW method, but may give different results on
		 * different machines due to varying roundoff behavior.  The integer methods should give the same results
		 * on all machines.  On machines with sufficiently fast FP hardware, the floating-point method may also be
		 * the fastest.  The IFAST method is considerably less accurate than the other two; its use is not
		 * recommended if high quality is a concern.  @{
		 */
		inline virtual JWCompress &use_integer_slow_dct( void )		{ cinfo.dct_method = JDCT_ISLOW; return *this; }
		inline virtual JWCompress &use_integer_fast_dct( void )		{ cinfo.dct_method = JDCT_IFAST; return *this; }
		inline virtual JWCompress &use_float_dct( void )			{ cinfo.dct_method = JDCT_FLOAT; return *this; }
		/// @}

		/// Determine if smoothing is accessed.  Zero turns it off, 100 sets the maximum amount of smoothing.
		inline virtual JWCompress &set_smoothing( const int smoothing = 0 ) { cinfo.smoothing_factor = std::max(0, std::min(100, smoothing)); return *this; }

		/// IJG's compression structure.  @see jpeglib.h
		jpeg_compress_struct cinfo;
};

