/* SimpleMessageKeeper (C) 2017 Stephane Charette <stephanecharette@gmail.com>
 * $Id: smk.cpp 2117 2017-01-18 13:18:22Z stephane $
 */


#include "smk.hpp"
#include <fstream>
#include <vector>
#include <set>
#include <cstdlib>
#include <sys/socket.h>
#include <netinet/in.h>
#include <unistd.h>


SMK::SMK( void ) :
	filename( "/opt/ccr/smk.json" ),
	s( -1 )
{
	/* C++ streams don't expose their file descriptors, so it is difficult to use file locking such as flock() and
	 * lockf().  And using mutexes and semaphores under Linux is problematic:
	 *			https://www.ccoderun.ca/programming/2010-01-13_PosixSemaphores/
	 * So instead, use a socket bound to a specific port as a file lock.  When the application finishes or is killed,
	 * the socket will automatically be freed, even if the application experiences a "kill -9".
	 */
	s = socket( AF_INET, SOCK_DGRAM, 0 );
	if (s < 0) throw std::runtime_error( "Failed to open a socket." );

	int retry = 5;
	while (true)
	{
		sockaddr_in addr = { AF_INET, htons(48782), htonl(INADDR_LOOPBACK), 0 };
		const int rc = bind( s, reinterpret_cast<sockaddr*>(&addr), sizeof(addr) );

		if (rc == 0) break;

		if (retry >= 0)
		{
			retry --;
			sleep(1);	// seconds
			continue;
		}

		throw std::runtime_error( "Failed to bind." );
	}

	return;
}


SMK::~SMK( void )
{
	close(s);

	return;
}


json SMK::load( void )
{
	std::ifstream i(filename);
	json j;
	i >> j;
	i.close();

	if (j.empty())
	{
		initialize_json(j);
	}

	return j;
}


SMK &SMK::save( json &j )
{
	if (j.empty())
	{
		initialize_json(j);
	}
	sort_messages_by_timestamp(j);
	j["smk"]["save_timestamp"] = std::time(nullptr);
	std::ofstream o(filename);
	o << std::setw(4) << j << std::endl;
	o.close();

	// sanity check -- make sure we're not filling up the hard drive!
	if (j["msg"].size() > 50)
	{
		clean();
	}
	
	return *this;
}


SMK &SMK::show_usage( const std::string &arg0 )
{
	std::cout	<< ""																<< std::endl
				<< arg0 << " <command> [<parameter>...]"							<< std::endl
				<< ""																<< std::endl
				<< "Commands and parameters:"										<< std::endl
				<< "\treset                        clear all messages"				<< std::endl
				<< "\tclean                        keep only a few recent messages"	<< std::endl
				<< "\tunique <priority> \"<text>\"   add a unique message"			<< std::endl
				<< "\tadd <priority> \"<text>\"      add or update a message"		<< std::endl
				<< "\tdel <id>|\"<text>\"            delete a message"				<< std::endl
				<< "\tget <id>|all|recent          display messages"				<< std::endl
				<< "\tshow                         pretty-print all messages"		<< std::endl
				<< "\tdump                         dump the json"					<< std::endl
				<< ""																<< std::endl
				<< "Priority in \"add\" and \"unique\" is:"							<< std::endl
				<< "\t0  = high"													<< std::endl
				<< "\t1  = medium"													<< std::endl
				<< "\t2  = low"														<< std::endl
				<< "\t3+ = other"													<< std::endl
				<< ""																<< std::endl
				<< "ID in \"del\" and \"get\" starts at 1."							<< std::endl
				<< ""																<< std::endl;

	return *this;
}


SMK &SMK::initialize_json( json &j )
{
	j["smk"]["reset_timestamp"]	= std::time(nullptr);
	j["smk"]["version"]			= SMK_VER;

	return *this;
}


SMK &SMK::reset( void )
{
	json j;
	initialize_json(j);

	return save(j);
}


SMK &SMK::clean( void )
{
	json j = load();

	// create a set (sorted from small to big) of message times
	std::set<std::time_t> time_of_messages;
	for (const auto &element : j["msg"])
	{
		const std::time_t tt = element["timestamp"];
		time_of_messages.insert( tt );
	}

	const std::time_t seconds_per_month		= 2628000;
	const std::time_t now					= std::time(nullptr);
	const std::time_t one_month_ago			= now - seconds_per_month;

	// copy messages until either of the following conditions are met:
	//		1) we have 10 messages
	//		2) the next message is more than a month old
	json old_msg;
	old_msg.swap( j["msg"] );
	for (const auto &tt : time_of_messages)
	{
		if (tt < one_month_ago || j["msg"].size() >= 10)
		{
			// the only thing remaining is messages older than 1 month old, or we already have 10 meesages
			break;
		}

		for (auto element : old_msg)
		{
			if (element["timestamp"] == tt)
			{
				element["count"] = 1;	// reset the count as if the older messages were deleted
				j["msg"].push_back( element );

				if (j["msg"].size() >= 10)
				{
					// we have enough messages, stop looking for more
					break;
				}
			}
		}
	}

	if (old_msg.size() != j["msg"].size())
	{
		// the messages have been modified, so we need to write out the new file
		save(j);
	}

	return *this;
}


SMK &SMK::sort_messages_by_timestamp( json &j )
{
	// re-order messages so the newer ones are at the bottom, and older ones at the top

	// create a set (sorted from small to big) of message times
	json old_msg;
	old_msg.swap( j["msg"] );
	std::set<std::time_t> time_of_messages;
	for (const auto &element : old_msg)
	{
		const std::time_t tt = element["timestamp"];
		time_of_messages.insert( tt );
	}

	// now that we have all the timestamp, go through them and add the necessary messages to "j"
	for ( auto	iter  = time_of_messages.begin();
				iter != time_of_messages.end();
				iter ++ )
	{
		const std::time_t tt = *iter;
		for (auto element : old_msg)
		{
			if (element["timestamp"] == tt)
			{
				j["msg"].push_back(element);
			}
		}
	}

	return *this;
}


SMK &SMK::add( const std::string &priority, const std::string &msg, const bool unique )
{
	json j;

	// try twice to load the .json file, trimming the file if necessary in between the two tries
	for (int loop=0; loop < 2; loop++)
	{
		j = load();

		if (j["msg"].size() < 20)
		{
			break;
		}

		// otherwise, we have too many messages!
		clean();
	}

	const std::time_t now = std::time(nullptr);

	// see if this message already exists before we add a new text string
	bool found = false;
	for (auto &element : j["msg"])
	{
		if (element["text"] == msg)
		{
			element["timestamp"	] = now;
			element["priority"	] = std::atoi(priority.c_str());

			if (unique)
			{
				// we only want 1 of these messages
				element["count"] = 1;
			}
			else
			{
				element["count"] = element["count"].get<int>() + 1;
			}

			found = true;
			break;
		}
	}

	if ( not found )
	{
		// add a new record into the JSON array
		json new_msg;
		new_msg["timestamp"	] = now;
		new_msg["priority"	] = std::atoi(priority.c_str());
		new_msg["text"		] = msg;
		new_msg["count"		] = 1;
		j["msg"].push_back(new_msg);
	}

	return save(j);

	return *this;
}


SMK &SMK::unique( const std::string &priority, const std::string &msg )
{
	return add( priority, msg, true );
}


SMK &SMK::del( const std::string &parm )
{
	json	j		= load();
	size_t	counter	= 0;
	
	const size_t idx = std::atoi(parm.c_str());
	if (idx)
	{
		if (idx > 0 && idx <= j["msg"].size())
		{
			/* When the indexes are shown to the users, they start at 1, and the list is shown in reverse where the
			 * newest messages are at the top of the list.  But in the json array, they start at zero and the array
			 * is ordered from oldest to newest.  So translate the index to something useful.
			 */
			const size_t fixed_index = j["msg"].size() - idx;
			j["msg"].erase(fixed_index);
			counter ++;
		}
	}
	else
	{
		// else if we get here we need to search for partial text matches

		auto iter = j["msg"].begin();
		while (iter != j["msg"].end())
		{
			const std::string &text = (*iter)["text"];
			if (text.find(parm) != std::string::npos)
			{
				// this entry needs to be deleted
				j["msg"].erase(iter);
				counter ++;

				// since we invalidated the iterator, start again from scratch
				iter = j["msg"].begin();
				continue;
			}

			// move on to the next message
			iter ++;
		}
	}

	if (counter)
	{
		save(j);
	}

	return *this;
}


SMK &SMK::dump( void )
{
	std::cout << load().dump(4) << std::endl;

	return *this;
}


std::string SMK::format( const std::time_t now, const std::time_t tt )
{
	const std::time_t second	= 1;
	const std::time_t minute	= 60	* second;
	const std::time_t hour		= 60	* minute;
	const std::time_t day		= 24	* hour;
	const std::time_t week		= 7		* day;
	const std::time_t year		= 365	* day;
	const std::time_t month		= year	/ 12;
	const std::time_t diff		= now	- tt;

	if (diff < 5	* second) return "now";
	if (diff < 2	* minute) return std::to_string(diff / second	) + " seconds ago";
	if (diff < 2	* hour	) return std::to_string(diff / minute	) + " minutes ago";
	if (diff < 2	* day	) return std::to_string(diff / hour		) + " hours ago";
	if (diff < 2	* week	) return std::to_string(diff / day		) + " days ago";
	if (diff < 2	* month	) return std::to_string(diff / week		) + " weeks ago";
	if (diff < 2	* year	) return std::to_string(diff / month	) + " months ago";

	return "long ago";
}


std::string SMK::format( const int priority )
{
	if (priority == 0)	return "high";
	if (priority == 1)	return "medium";
	if (priority == 2)	return "low";

	return "other";
}


SMK &SMK::format( void )
{
	json j = load();

	if (j["msg"].empty() == false)
	{
		const std::time_t now( std::time(nullptr) );

		/* ANSI colours:  https://en.wikipedia.org/wiki/ANSI_escape_code#Colors
		 * 
		 *	 0 = reset colors back to default
		 *	 1 = turn brightness on
		 *	30 = black
		 *	31 = red
		 *	32 = green
		 *	33 = yellow
		 *	34 = blue
		 *	35 = magenta
		 *	36 = cyan
		 *	37 = white
		 */

//		std::cout << "\e[1;30m_____ When ____ Priority  ____ Message _____\e[0m" << std::endl;

		for (auto	iter  = j["msg"].rbegin();
					iter != j["msg"].rend();
					iter ++ )
		{
			const auto			element			= *iter;
			const std::string	when			= format(now, element["timestamp"]);
			const std::string	priority		= format(element["priority"]);
			const std::string	priority_colour	= (priority == "high" ? "\e[1;31m" : "\e[1;32m");
			const int			count			= element["count"];
			const std::string	count_text		= (count < 2 ? "" : "\e[1;31m" + std::to_string(count) + " x ");
			const std::string	text			= element["text"];

			std::cout
					<< "\e[1;37m"		<< std::right	<< std::setw(15) << when
					<< "\e[0;32m"		<< std::left	<< " ["
					<< priority_colour	<< std::left	<< std::setw( 7) << priority
					<< "\e[0;32m"		<< std::left	<< "] "
					<< count_text
					<< "\e[1;33m"		<< std::left	<< text
					<< "\e[0m"
					<< std::endl;
		}
	}

	return *this;
}


SMK &SMK::get_all( void )
{
	return display_msg( load()["msg"] );
}


SMK &SMK::get_recent( void )
{
	json j = load();
	json msg;

	const std::time_t now( std::time(nullptr) );

	//											hour  day    week    month    year
	//											|     |      |       |        |
	const std::vector<std::time_t> seconds	= { 3600, 86400, 604800, 2628000, 31536000 };
	for (const auto tt : seconds)
	{
		// see if we have any messages within the past "x" seconds
		const std::time_t time_limit = now - tt;

		for (const auto &element : j["msg"])
		{
			if (element["timestamp"] >= time_limit)
			{
				// this message is within the past "x" seconds
				msg.push_back(element);
			}
		}

		// if we've found something, stop looking
		if (msg.size() > 0)
		{
			break;
		}
	}

	if (msg.empty() == false)
	{
		display_msg(msg);
	}

	return *this;
}


SMK &SMK::get( const std::string &parm )
{
	json j = load();

	const size_t idx		= std::atoi( parm.c_str() );
	const size_t max_idx	= j["msg"].size();

	if (idx <= max_idx)
	{
		display_msg( j["msg"][max_idx - idx] );
	}

	return *this;
}


SMK &SMK::display_msg( const json &j )
{
	if (j.empty())
	{
		// do nothing
	}
	else if (j.is_array())
	{
		size_t idx = 0;
		for (auto	iter  = j.rbegin();
					iter != j.rend();
					iter ++ )
		{
			idx ++;
			std::cout << idx << "\t";
			display_msg( *iter );
		}
	}
	else
	{
		const std::string text = j["text"];	// avoid json's operator<<() from printing double quotes around the string
		std::cout << j["timestamp"] << "\t" << j["priority"] << "\t" << j["count"] << "\t" << text << std::endl;
	}

	return *this;
}
