/* SimpleMessageKeeper (C) 2017 Stephane Charette <stephanecharette@gmail.com>
 * $Id: smk.hpp 2112 2017-01-18 06:22:54Z stephane $
 */


#pragma once

#include <string>
#include <ctime>
#include "json.hpp"
using json = nlohmann::json;


/// Class to implement reading, writing, adding, deleting, and formatting messages from the SMK @p .json file.
class SMK final
{
	public:

		/// Constructor.
		SMK( void );

		/// Destructor.
		~SMK( void );

		/// Load the json from @ref filename.
		json load( void );

		/// Save the json to @ref filename.
		SMK &save( json &j );

		/// Show a block of text that describes how to use SMK.
		SMK &show_usage( const std::string &arg0 );

		/// Erase the json in @ref filename and start with a brand new json structure.
		SMK &reset( void );

		/// Clean up old entries in @ref filename.  Will only keep the last few recent messages.
		SMK &clean( void );

		/** Add or update a message in @ref filename.  If the message already exists, then the count and timestamp
		 * are both updated.  If @p unique is set to @p true, then the count is set to @p 1 even if the message
		 * already existed.  @see @ref unique()
		 */
		SMK &add( const std::string &priority, const std::string &msg, const bool unique=false );

		/// Same thing as @ref add() but sets the @p unique parameter to @p true.  @see @ref add()
		SMK &unique( const std::string &priority, const std::string &msg );

		/** Delete a message from the json in @ref filename.  If @p parm is numeric, then it is assumed to be an index
		 * into the message array.  If @p parm is text, then a case-sensitive partial substring match will be done on
		 * all of the messages, and those found to match will be deleted.
		 */
		SMK &del( const std::string &parm );

		/// Dump the full json in @ref filename to stdout.
		SMK &dump( void );

		/// Format all the message to an easy-to-read block of text.  This can be used as a login message.
		SMK &format( void );

		/// Get a tab-delimited set of all messages.
		SMK &get_all( void );

		/// Get a tab-delimited set of recent messages.
		SMK &get_recent( void );

		/// Get a specific message from @ref filename.  The @p parm must be a valid numerical index, starting at @p 1.
		SMK &get( const std::string &parm );

	protected:

		/// Insert a few housekeeping items into the json.
		SMK &initialize_json( json &j );

		/** Ensure the messages in @ref filename are sorted in order from oldest to newest.  This is automatically
		 * called when the json file is saved to disk.
		 */
		SMK &sort_messages_by_timestamp( json &j );

		/// Format the given timestamp as an easy-to-read text string, such as @p "35 seconds ago" or @p "2 days ago".
		std::string format( const std::time_t now, const std::time_t tt );

		/// Format the numeric priority index to a text string, such as @p high or @p low.
		std::string format( const int priority );

		/** Format the given message or messages as a tab-delimited block of text.
		 * @see @ref get_all() @see @ref get_recent()
		 */
		SMK &display_msg( const json &j );

		/// The location where the json structure is stored.  This file must have read/write permissions.
		std::string filename;

		/** Mutex and semaphores are badly broken on Linux, and file locking not accessible from C++ streams.
		 * Instead, use a socket bound to a specific port to emulate file or process locking.
		 */
		int s;
};
