// TinyAES++ -- AES encryption/decryption for C and C++
// Public domain; see http://unlicense.org/
// Stephane Charette <stephanecharette@gmail.com>
// $Id: TinyAES.hpp 2438 2018-01-23 21:45:08Z stephane $

#pragma once

/*
 * ******************************************************
 * Instead of trying to read through the Doxygen markup in
 * this file, refer to the official documentation instead:
 * https://www.ccoderun.ca/programming/doxygen/tinyaes++/
 * *******************************************************
 */

#include <string>
#include <vector>
#include <cstdint>


/** %TinyAES provides a few variants of the encryption/decryption methods in
 * the AES family.  Only 256-bit AES is enabled in this library, and only a
 * few modes are available:
 *
 * @li Electronic Codebook (ECB) should @b not be used.
 * See the documentation on ECB here:  https://en.wikipedia.org/wiki/Block_cipher_mode_of_operation#Electronic_Codebook_.28ECB.29
 *
 * @li Cipher Block Chaining (CBC) is good to use.  See the documentation on CBC here:  https://en.wikipedia.org/wiki/Block_cipher_mode_of_operation#Cipher_Block_Chaining_.28CBC.29
 *
 * @li Integer Counter Mode (CTR) is good to use.  See the documentation on CTR here:  https://en.wikipedia.org/wiki/Block_cipher_mode_of_operation#Counter_.28CTR.29
 *
 * @note The C++ wrapper only exposes 256-bit AES in CBC mode!  No other bit
 * size, and no other mode is enabled if you are using the TinyAES namespace
 * from within your C++ application.
 *
 * You need 3 things to encrypt and decrypt when using 256-bit AES in CBC mode:
 * @li a 256-bit key, meaning 32 bytes of random data
 * @li a 128-bit initialization vector, meaning 16 bytes of random data
 * @li a text string or binary data that you want to encrypt and/or decrypt
 *
 * The key and the iv (initialization vector) are required to correctly
 * decrypt what was previously encrypted.  If you don't have the key and the
 * iv, you @b cannot decrypt.  Attempting to guess the key and the iv is
 * unlikely:  with a 256-bit key, it would take a supercomputer trillions of
 * years to crack the key via brute force exhaustion of the key space.
 *
 * This library can help with the following tasks:
 *
 * @li create a random key and random iv:  @ref TinyAES::generate_random_key_and_iv()
 * @li convert a key and iv from a vector of bytes to a text string for easy storage:  @ref TinyAES::to_hex_string()
 * @li convert a text string back to a key and iv:  @ref TinyAES::from_hex_string()
 * @li encrypt a text string (@p std::string):  @ref TinyAES::cbc_encrypt()
 * @li encrypt a vector of bytes (@ref TinyAES::VBytes):  @ref TinyAES::cbc_encrypt()
 * @li decrypt a text string:  @ref TinyAES::cbc_decrypt()
 * @li decrypt a vector of bytes:  @ref TinyAES::cbc_decrypt()
 *
 * The encrypt/decrypt functions that take and return a @p std::string instead
 * of a vector of bytes were added to make life easy for developers who happen
 * to have some @p std::string objects they want encrypted.  The API converts
 * the strings to @ref TinyAES::VBytes objects, and then converts the results
 * back to @p std::string.
 *
 * If wondering whether you should use the @p std::string or
 * @ref TinyAES::VBytes function calls, prefering @ref TinyAES::VBytes will
 * save two extra copies of the data -- one on input, and the other on output.
 *
 * In the simplest case, this is how you'd encrypt and decrypt a string of text:
 * ~~~~
 * #include <TinyAES.hpp>
 *
 * // ...
 *
 * TinyAES::VBytes key;
 * TinyAES::VBytes iv;
 * TinyAES::generate_random_key_and_iv(key, iv);
 *
 * std::string original_string = "Hello, World!";
 * std::string encrypted_string = TinyAES::cbc_encrypt(original_string, key, iv);
 *
 * // ...
 *
 * std::string decrypted_string = TinyAES::cbc_decrypt(encrypted_string, key, iv);
 * ~~~~
 * If you need to store and restore the 32-byte key and 16-byte initialization
 * vector, this can easily be done using @ref TinyAES::to_hex_string() and
 * @ref TinyAES::from_hex_string().
 *
 * It is up to the application or the end user to ensure the key and iv are
 * either copied or moved to an appropriate location to decrypt.  There are no
 * magical functions within TinyAES which will move the key and iv to their
 * required final destination.  And you cannot simply encrypt the key and the
 * iv, since decryption requires you to have the key and iv!
 */
namespace TinyAES
{
	/** A vector of bytes.  Used to pass in the key, the initialization vector,
	 * and the byte to encrypt/decrypt.
	 *
	 * ~~~~
	 * #include <TinyAES.hpp>
	 *
	 * using Vector_of_bytes=TinyAES::VBytes;
	 *
	 * Vector_of_bytes get_iv()
	 * {
	 * 	// create and return a 128-bit (16 byte) initialization vector
	 * 	Vector_of_bytes v =
	 * 	{
	 * 		// poor choice for an iv (consecutive bytes, perhaps easy to guess?)
	 * 		// but shows how to easily instantiate a vector of bytes;
	 * 		// [instead, see TinyAES::generate_random_key_and_iv()]
	 * 		0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
	 * 		0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f
	 * 	};
	 * 	return v;
	 * }
	 *
	 * Vector_of_bytes get_key()
	 * {
	 * 	// create and return a 256-bit (32 byte) encryption/decryption key
	 * 	Vector_of_bytes v =
	 * 	{
	 * 		// same comment as above, non-random key would still encrypt/decrypt
	 * 		// as expected, but attackers might have an easy time guessing the
	 * 		// key making it a trivial task to decrypt your data
	 * 		// [instead, see TinyAES::generate_random_key_and_iv()]
	 * 		0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
	 * 		0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
	 * 		0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
	 * 		0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f
	 * 	};
	 * 	return v;
	 * }
	 * ~~~~
	 */
	typedef std::vector<uint8_t> VBytes;

	/** The number of bytes in a key, which is 32 for 256-bit AES.
	 * @see @ref generate_random_key_and_iv()
	 */
	const size_t key_size_in_bytes = 32;

	/** The number of bytes in the initialization vector.  (Normally the
	 * iv is the same size as a block, which for AES is always 16 bytes.)
	 * @see @ref generate_random_key_and_iv()
	 */
	const size_t iv_size_in_bytes = 16;

	/** Used internally to validate the length of both the key and the
	 * initialization vector.  @note Under normal circumstances, user
	 * code has no need to make explicit calls to this function.
	 */
	void validate_key_and_iv(const VBytes &key, const VBytes &iv);

	/** Generate a random 256-bit key and a 128-bit initialization vector.
	 * These will need to be stored somewhere, since decrypting
	 * requires the @b same key and iv as was used during encryption.
	 *
	 * ~~~~
	 * TinyAES::VBytes key;
	 * TinyAES::VBytes iv;
	 * 
	 * if (previous_app_settings_exist)
	 * {
	 * 	key	= load_encryption_key();
	 * 	iv	= load_encryption_iv();
	 * 
	 * }
	 * 
	 * if (key.size()	!= TinyAES::key_size_in_bytes	||
	 * 	iv.size()	!= TinyAES::iv_size_in_bytes	)
	 * {
	 * 	// we *must* have a valid key and iv to encrypt/decrypt
	 * 	TinyAES::generate_random_key_and_iv(key, iv);
	 * 	save_encryption_key(key);
	 * 	save_encryption_iv(iv);
	 * }
	 * ~~~~
	 *
	 * @param [out] key Any previous content will be completely overwritten.
	 * On output, the key will contain exactly 256 bits (32 bytes).
	 *
	 * @param [out] iv Any previous content will be completely overwritten.  On
	 * output, the initialization vector will contain exactly 128 bits (16 bytes).
	 *
	 * @param [in] seed If left as zero, a seed based on the current time will
	 * be used.  If set to anything other than zero, that value will be used
	 * as a seed.
	 */
	void generate_random_key_and_iv(VBytes &key, VBytes &iv, size_t seed = 0);

	/** Convert the vector into a text string of hex characters.  Useful in
	 * converting the encryption key and initialization vector to a text
	 * string.
	 *
	 * ~~~~
	 * TinyAES::VBytes key;
	 * TinyAES::VBytes iv;
	 * 
	 * TinyAES::generate_random_key_and_iv(key, iv));
	 * 
	 * std::cout << "key="	<< TinyAES::to_hex_string(key)	<< std::endl;
	 * std::cout << "iv="	<< TinyAES::to_hex_string(iv)	<< std::endl;
	 * ~~~~
	 * The output would look similar to this:
	 * ~~~~{.txt}
	 * key=ba531427ce62454abcbdfd37a247593d9dc141ccce55f4f978e831763325decd
	 * iv=67bc931c150804b1ba2152b01b368b26
	 * ~~~~
	 * Note how the key is twice as long as the initialization vector, since
	 * the key contains 256 bits (32 bytes) of data and the iv contains 128
	 * bits (16 bytes) of data.
	 *
	 * @see @ref from_hex_string()
	 */
	std::string to_hex_string(const VBytes &v);

	/** Convert the text string back into a vector of bytes.  Useful in
	 * converting textual representations of the encryption key and iv back
	 * into a vector of bytes.
	 *
	 * ~~~~
	 * TinyAES::VBytes key	= TinyAES::from_hex_string("ba531427ce62454abcbdfd37a247593d9dc141ccce55f4f978e831763325decd"); 
	 * TinyAES::VBytes iv	= TinyAES::from_hex_string("67bc931c150804b1ba2152b01b368b26");
	 * ~~~~
	 *
	 * @see @ref to_hex_string()
	 */
	VBytes from_hex_string(const std::string &str);

	/** Encrypt the input buffer using 256-bit AES Cipher Block Chaining mode.
	 *
	 * Similar to the other @ref TinyAES::cbc_encrypt() method, but takes a
	 * @p std::string instead of a vector of bytes.
	 *
	 * Automatically handles the PKCS padding internally, so the output string
	 * will always be between 1 and 16 bytes larger than the input string.
	 * This padding will automatically be removed when
	 * @ref TinyAES::cbc_decrypt() is called, so other than knowing and
	 * expecting the encrypted string to be a few bytes bigger than the
	 * original text, there should be no user impact.
	 *
	 * ~~~~
	 * std::string encrypt(const std::string &text)
	 * {
	 * 	const TinyAES::VBytes key	= get_key();
	 * 	const TinyAES::VBytes iv	= get_iv();
	 * 	
	 * 	const std::string encrypted = TinyAES::cbc_encrypt(text, key, iv);
	 * 	
	 * 	return encrypted;
	 * }
	 * ~~~~
	 *
	 * @param [in] str While a @p std::string normally implies "text", it
	 * can contain anything, including binary non-textual data.
	 *
	 * @param [in] key The encryption key must be 256 bits in length (32 bytes).
	 *
	 * @param [in] iv The initialization vector must be 128 bits in length (16 bytes).
	 */
	std::string cbc_encrypt(const std::string &str, const VBytes &key, const VBytes &iv);

	/** Encrypt the input buffer using 256-bit AES Cipher Block Chaining mode.
	 *
	 * Similar to the other @ref cbc_encrypt() method, but takes and returns a
	 * vector of bytes instead of a @p std::string.
	 *
	 * Automatically handles the PKCS padding internally.  The output vector
	 * will always be between 1 and 16 bytes larger than the input vector.
	 * This padding will automatically be removed when
	 * @ref TinyAES::cbc_decrypt() is called, so other than knowing and
	 * expecting the encrypted string to be a few bytes bigger than the
	 * original text, there should be no user impact.
	 *
	 * ~~~~
	 * TinyAES::VBytes encrypt(const TinyAES::VBytes &input)
	 * {
	 * 	const TinyAES::VBytes key		= get_key();
	 * 	const TinyAES::VBytes iv		= get_iv();
	 * 	
	 * 	const TinyAES::VBytes encrypted	= TinyAES::cbc_encrypt(input, key, iv);
	 * 	
	 * 	return encrypted;
	 * }
	 * ~~~~
	 *
	 * @param [in] input The vector of bytes to encrypt.  Can be any length.
	 *
	 * @param [in] key The encryption key must be 256 bits in length (32 bytes).
	 *
	 * @param [in] iv The initialization vector must be 128 bits in length (16 bytes).
	 */
	VBytes cbc_encrypt(const VBytes &input, const VBytes &key, const VBytes &iv);

	/** Decrypt the buffer using 256-bit AES Cipher Block Chaining mode.
	 *
	 * Automatically handles the PKCS padding internally.  The output string
	 * will always be between 1 and 16 bytes shorter than the encrypted input
	 * due to the removal of the padding bytes.
	 *
	 * ~~~~
	 * std::string decrypt()
	 * {
	 * 	const TinyAES::VBytes key	= get_key();
	 * 	const TinyAES::VBytes iv	= get_iv();
	 * 	const std::string encrypted	= get_encrypted_text();
	 * 	
	 * 	const std::string decrypted = TinyAES::cbc_decrypt(encrypted, key, iv);
	 * 	
	 * 	return decrypted;
	 * }
	 * ~~~~
	 *
	 * @param [in] input The encrypted input must be at least 16 bytes in length,
	 * and will be a muliple of 16 (since AES works on 16-byte blocks).
	 *
	 * @param [in] key The encryption key must be 256 bits in length (32 bytes).
	 *
	 * @param [in] iv The initialization vector must be 128 bits in length (16 bytes).
	 *
	 * @see @ref cbc_encrypt()
	 */
	std::string cbc_decrypt(const std::string &input, const VBytes &key, const VBytes &iv);

	/** Decrypt the input buffer using 256-bit AES Cipher Block Chaining mode.
	 *
	 * Similar to the other @ref cbc_decrypt() method, but takes a vector of
	 * bytes instead of a @p std::string.
	 *
	 * Automatically handles the PKCS padding internally.  The output vector
	 * will always be between 1 and 16 bytes shorter than the encrypted input
	 * due to the removal of the padding bytes.
	 *
	 * ~~~~
	 * TinyAES::VBytes decrypt()
	 * {
	 * 	const TinyAES::VBytes key		= get_key();
	 * 	const TinyAES::VBytes iv		= get_iv();
	 * 	const TinyAES::VBytes encrypted	= get_encrypted_text();
	 * 	
	 * 	const TinyAES::VBytes decrypted	= TinyAES::cbc_decrypt(encrypted, key, iv);
	 * 	
	 * 	return decrypted;
	 * }
	 * ~~~~
	 *
	 * @param [in] input The encrypted input must be at least 16 bytes in length,
	 * and will be a muliple of 16 (since AES works on 16-byte blocks).
	 *
	 * @param [in] key The encryption key must be 256 bits in length (32 bytes).
	 *
	 * @param [in] iv The initialization vector must be 128 bits in length (16 bytes).
	 *
	 * @see @ref cbc_encrypt()
	 */
	VBytes cbc_decrypt(const VBytes &input, const VBytes &key, const VBytes &iv);
}
