// TinyAES++ -- AES encryption/decryption for C and C++
// Public domain; see http://unlicense.org/
// Stephane Charette <stephanecharette@gmail.com>
// $Id: test_key_and_iv.cpp 2416 2017-12-04 23:44:48Z stephane $

#include "TinyAES.hpp"
#include <gtest/gtest.h>


TEST(KeyAndIVLen, ZeroLen)
{
	TinyAES::VBytes key;
	TinyAES::VBytes iv;

	ASSERT_EQ(key.size(), 0);
	ASSERT_EQ(iv.size(), 0);

	ASSERT_THROW( TinyAES::validate_key_and_iv(key, iv), std::exception );

    return;
}


TEST(KeyAndIVLen, NonZeroLen)
{
	ASSERT_EQ( TinyAES::key_size_in_bytes, 32 );
	ASSERT_EQ( TinyAES::iv_size_in_bytes, 16 );

	TinyAES::VBytes key;
	TinyAES::VBytes iv;
	ASSERT_NO_THROW(TinyAES::generate_random_key_and_iv(key, iv));

	ASSERT_EQ(key.size(), 32);
	ASSERT_EQ(iv.size(), 16);
	ASSERT_EQ(key.size(), TinyAES::key_size_in_bytes);
	ASSERT_EQ(iv.size(), TinyAES::iv_size_in_bytes);

	ASSERT_NO_THROW(TinyAES::validate_key_and_iv(key, iv));
}


TEST(KeyAndIVLen, ConvertToText)
{
	TinyAES::VBytes key;
	TinyAES::VBytes iv;

	ASSERT_NO_THROW(TinyAES::generate_random_key_and_iv(key, iv));
	ASSERT_NO_THROW(TinyAES::validate_key_and_iv(key, iv));

	const std::string key_string = TinyAES::to_hex_string(key);
	ASSERT_EQ(key_string.size(), 64);

	TinyAES::VBytes v = TinyAES::from_hex_string(key_string);
	ASSERT_EQ(v.size(), TinyAES::key_size_in_bytes);
	ASSERT_EQ(v, key);
}


TEST(KeyAndIV, ToHexString)
{
	const TinyAES::VBytes v =
	{
		0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09,
		0xf6, 0xf7, 0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xfe, 0xff
	};

	const std::string str = TinyAES::to_hex_string(v);

	ASSERT_EQ(str, "00010203040506070809f6f7f8f9fafbfcfdfeff");
}


TEST(KeyAndIV, FromHexString)
{
	const std::string str = "00010203040506070809f6f7f8f9fafbfcfdfeff";

	const TinyAES::VBytes v1 = TinyAES::from_hex_string(str);
	const TinyAES::VBytes v2 =
	{
		0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09,
		0xf6, 0xf7, 0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xfe, 0xff
	};

	ASSERT_EQ(v1, v2);
}


TEST(KeyAndIVLen, DuplicateKeyAndIV)
{
	TinyAES::VBytes key1;
	TinyAES::VBytes key2;
	TinyAES::VBytes iv1;
	TinyAES::VBytes iv2;

	TinyAES::generate_random_key_and_iv(key1, iv1);
	TinyAES::generate_random_key_and_iv(key2, iv2);

	TinyAES::validate_key_and_iv(key1, iv1);
	TinyAES::validate_key_and_iv(key2, iv2);

	std::cout << "key1=" << TinyAES::to_hex_string(key1) << std::endl;
	std::cout << "key2=" << TinyAES::to_hex_string(key2) << std::endl;

	std::cout << "iv1=" << TinyAES::to_hex_string(iv1) << std::endl;
	std::cout << "iv2=" << TinyAES::to_hex_string(iv2) << std::endl;

	ASSERT_NE(key1, key2);
	ASSERT_NE(iv1, iv2);
}
